local VERSION = "3.3"

--[[
SCRIPT: adptr3.lua
VERSION: 3.3
AUTHOR: Mike Shellim
URL: http://www.rc-soar.com/opentx/lua

DESCRIPTION
===========
Script for crow-aware adaptive elevator trim
Dynamically modifies 5-point crow->elevator ('compensation') curve in response to trim clicks.

REQUIREMENTS
============
Any OpenTX or EdgeTX transmitter
OpenTX 2.2.2 or above, with Lua build option
EdgeTX 2.4 or above

DOCUMENTATION
=============
!! READ THE DOCUMENTATION BEFORE INSTALLING !!
https://rc-soar.com/opentx/lua/adaptivetrim/AdaptiveTrim_v3.3.pdf

HISTORY
=======
23/01/26  v3.3 Fix: detection of widget mode (colour screen radios)
10/05/23  V3.2 Added support for base 1 curve tables (new function getCurve()), removed os checks
10/12/20  V3.1 fixed occasional "syntax error" (added forward declaration of drawLine)
30/08/20  v3.0 forked from adpt22.lua (v 2.2.1). Now with common codebase for widget and telemetry scripts.

SAFETY
======
Test carefully before flight. 
Use at own risk.
IF IN DOUBT DO NOT FLY!!
--]]


local last_error	-- last error

-- Input/output limits
local XMAX = 1024 -- max crow input
local XMIN = -1024 -- min crow input
local YMAX = 100 -- max elevator output
local YMIN = -100 -- min elevator output

-- Sounds
local BEEP_HI = 1900
local BEEP_LO = 900
local BEEP_DUR = 90

-- Crow/elev compensation curve
local COMP_CV_NAME = "adp"
local comp_cv_idx
local pt_x = {-XMAX, -XMAX/2, 0, XMAX/2, XMAX}
local x_label
local idle_crow_limit

-- Script parameters from curve 'prm' 
local crow_fm 	-- flight mode in which crow is active
local crow_chan_id	-- crow channel id
local idle_mode	-- idle crow mode
local MODE_EMUDFLT = 0 -- emulate system behaviour
local MODE_PINTOZERO = 1 -- pin to zero

-- Trim state
local trim_source -- GV (>0)  or LS (<0) storing trim state
local state -- idle, clicked, repeat, wait_long
local IDLE = 1
local CLICKED = 2
local WAIT_LONG = 4
local wait = {0, 27, 9, 72} -- wait duration (10ms units) indexed by state. 
local t0 -- time of last click


-- Crow range is divided into 7 regions. Each region defines one or more points to move
-- xMax: upper boundary of region
-- pts: set of active points
local regions = {
	{xMax=-925, pts={[1]=true}}, -- max crow
	{xMax=-455, pts={[2]=true}},
	{xMax=-185, pts={[2]=true, [3]=true}},
	{xMax=185, pts={[3]=true}},
	{xMax=455, pts={[3]=true, [4]=true}},
	{xMax=925, pts={[4]=true}},
	{xMax=1024, pts={[5]=true}} -- zero crow region. One point max.
	}

local NPTS = 5 -- expected number of points in compensation curve.
local IS_COLOR = ( LCD_W == 480 )
local IS_WGT = ... -- set to true if called as a widget

local font = SMLSIZE
local line_ht

-----------------
-- WIDGET OPTIONS
-----------------
local options = {}


----------------------------
-- CALLED ONCE TO INITIALISE
----------------------------
local getCurveIdxFromName
local getCurve
local function init()
  last_error = nil
  state = IDLE
  t0 = 0

	-- Removed os checks since earlier getVersion in earlier versions of ETX and OTX did not support osname
  -- local _, _, maj, minor, rev = getVersion()
  -- if 10000*maj + 100*minor + rev < 20202 then last_error = "Unsupported O/S ver" return end

	-- Get parameters from 'prm' curve
	local prm_cv_idx = getCurveIdxFromName  ('prm')
	if not prm_cv_idx then last_error = "Curve 'prm' not found" return end
	local cv = getCurve (prm_cv_idx)
  if cv == nil then last_error = "Curve 'prm' read error" return end
	if cv.points < 5 then last_error = "Curve 'prm' has < 5 pts"	return end

	-- Break out params and validate
	local params = cv.y
	if params[1] < 0 or params[1] > 8 then last_error = "Prm 1: Invalid FM number"	return	end
	crow_fm = params[1]

	-- Trim state from GV or LS?
	if	(params[2] >=1 and params[2] <= 9) then
		-- trim state supplied by GV
		trim_source = params[2] - 1
	elseif (params[2] >=-63 and params[2] <= -1) then
		-- trim state supplied by consecutive logical switches. Store id of first LS as negative integer.
		trim_source = -(-params[2] + getFieldInfo ("ls1").id - 1)
	else
		last_error = "Prm 2: Invalid CH or GV"
		return
	end

	-- Store id of crow channel
	if params[4] < 1 or params[4] > 32 then last_error = "Prm 4: chan # out of range" return end
	crow_chan_id = params[4] + getFieldInfo ("ch1").id - 1

	-- Get idle-crow mode
	if params[5] < 0 then last_error = "Prm 5: Invalid idle-crow mode" return end
	idle_mode = params[5]

	-- Look for crow-elevator compensation curve
	comp_cv_idx = getCurveIdxFromName (COMP_CV_NAME)
	if not comp_cv_idx then last_error = "CV:"..COMP_CV_NAME .." not found" return end
	cv = model.getCurve (comp_cv_idx)
  if cv == nil then last_error = "CV:"..COMP_CV_NAME .." read error" return end
	if cv.points ~= NPTS then last_error = "CV:"..COMP_CV_NAME .. " must have " .. NPTS .." pts" end
  if cv.type ~= 0 then last_error = "CV:"..COMP_CV_NAME .. " must be type 'simple' " end

	-- x-axis label
	x_label = model.getOutput (params[4]-1).name
	if not x_label or x_label == "" then
		x_label = "ch:" .. params[4]
	end
  
  -- Trim limits at zero crow
  if idle_mode == MODE_EMUDFLT then idle_crow_limit = 25
  elseif idle_mode == MODE_PINTOZERO then idle_crow_limit = 0 
  else idle_crow_limit = YMAX
  end

  -- set line height
  line_ht = IS_COLOR and 17 or 8 -- default if no lcd.sizeText function
  if lcd.sizeText then
    _, line_ht = lcd.sizeText ("X", font)
  end
end

-- Called when a widget is created
----------------------------------
local function create(zone, options)
	init()
  return {zone=zone, options=options}
end

-- Called when widget options have changed
------------------------------------------
local function update(widget, options)
    widget.options = options
end


-- Get crow value and region number
-----------------------------------
local function getCrowAndRegion ()
  -- Get crow value and clip to bounds
	local cr = getValue (crow_chan_id)
  cr = math.max (math.min (cr, XMAX), XMIN)
  
  local r = 1
  while regions[r].xMax < cr do
    r = r + 1
  end
  return cr, r
end

-- Determine amount to adjust by from trim state (GVAR or LS)
local function getTrimState()
  local delta = 0
	if trim_source > 0 then
		delta = model.getGlobalVariable (trim_source, crow_fm)
	else
    -- <0 so LS
		if getValue (-trim_source) > 0 then delta = -1
		elseif getValue (-trim_source + 1) > 0 then delta = 1
		end
	end
  return delta
end


-----------------------------------------------------------------------------
-- CALLED PERIODICALLY for a telemetry script regardless of screen visibility
-----------------------------------------------------------------------------
local function bg_func ()
  
  local t = getTime ()
  if t < (t0 + wait[state]) then 
    return end
  
  if (getFlightMode() ~= crow_fm) or last_error then
    state = IDLE
		return
	end
  
  local delta = getTrimState ()
  if delta == 0 then
    -- trim in middle
    state = IDLE
		return
  end
  
		-- Get the compensation curve
	local cv = getCurve (comp_cv_idx)
---@diagnostic disable-next-line: need-check-nil
	local pt_y = cv.y
  
  -- Calculate new y-values and set flag if changed
  local _, rgn  = getCrowAndRegion ()
  local ylim = (rgn == #regions) and idle_crow_limit or YMAX
	local is_cv_changed = false
  for i, _ in pairs (regions[rgn].pts) do
    local new_y = math.max (math.min (pt_y[i] + delta, ylim), -ylim)
		if pt_y[i] ~= new_y then
      pt_y[i] = new_y
			is_cv_changed = true
		end
	end

	-- Update curve
	if is_cv_changed then
---@diagnostic disable-next-line: need-check-nil
    model.setCurve (comp_cv_idx, {name=cv.name, smooth=1, y=pt_y})
  end

  -- promote click state (waiting->clicked, clicked->rpt)
  state = math.min (state + 1, 3)
  
  -- handle beeps
  if (rgn == #regions) and (idle_mode == MODE_EMUDFLT) then
    -- emulate default trim behaviour
    local y = pt_y[NPTS]
    if y == 0 and is_cv_changed then
      playFile ("system/midtrim.wav")
      state = WAIT_LONG
    elseif y == ylim and (is_cv_changed or state == CLICKED) then
      playFile ("system/maxtrim.wav")
    elseif y == -ylim and (is_cv_changed or state == CLICKED) then
      playFile ("system/mintrim.wav")
    elseif is_cv_changed then
      playTone (y/25*(BEEP_HI-BEEP_LO) + BEEP_HI, BEEP_DUR,0,PLAY_NOW)
    end
  else
    if is_cv_changed then 
        playTone (BEEP_HI, BEEP_DUR, 0, PLAY_NOW) 
    end
	end
  
  t0 = t
end

-- Called periodically for a widget when screen NOT visible
-----------------------------------------------------------
local function background(widget)
  bg_func ()
end

---------------------------
-- FUNCTION TO REFRESH SCREEN
-- and HANDLE KEY PRESSES
---------------------------
local t_next_blink = 0
local is_blink = false

local drawLine  -- v3.1 fix
local drawText  -- v3.1 fix

local drawPoint, sx, sy
local xorg, yorg
local width, height
local x_scale
local y_scale

local function run_func (widget)
  
  if IS_WGT then
    -- Keep responsive
    bg_func ()
    -- local widget = _
   	width = widget.zone.w - 8
    height = widget.zone.h - 4
    xorg = widget.zone.x + 4
    yorg = widget.zone.y + 2
  else
    width = LCD_W - 4
    height = LCD_H -    2 
    xorg = 2
    yorg = 1
    lcd.clear()
  end
  
  -- Scale from logical to device coordinates
	x_scale = (width)/(XMAX-XMIN+1)
	y_scale = (height)/(YMAX-YMIN+1)

	-- draw error message top left
	if last_error then
		drawText (xorg, yorg, "Adptrm error")
		drawText (xorg, yorg + line_ht, last_error)
		return
	end
    
	-- draw heading
	drawText (xorg + 2, yorg, x_label.."->elev (CV:"..COMP_CV_NAME .. ")")

	-- draw axes
	drawLine (XMAX, YMIN, XMAX, YMAX, SOLID)
	drawLine (XMIN, 0, XMAX, 0, SOLID)
  
  -- update blink state
	local t = getTime()
	if t > t_next_blink then
		t_next_blink = t + 20
		is_blink = not is_blink
	end
  
  -- read curve
  local cv = getCurve (comp_cv_idx)
---@diagnostic disable-next-line: need-check-nil
  local pt_y = cv.y -- base 1

  -- Get crow value and region
	local xSrc, rgn  = getCrowAndRegion ()

  -- Is the crow flight mode active?
  local fActive = (getFlightMode() == crow_fm)
  if not fActive then
    
    drawText (xorg+4, yorg + line_ht, "<inactive>")
    
  else
    -- draw captions on mono screens, and colour screen if zone has sufficient height 
    if not IS_COLOR or height > 80 then
      drawText (xorg + width - 10, sy(0) + 2, x_label, RIGHT )
      drawText (xorg+4, yorg + line_ht,  "Trim down")
      drawText (xorg+4, yorg + height - 2*line_ht, "Trim up")
    end
    
    -- zero-crow trim limits
    if idle_mode == MODE_EMUDFLT then
      drawLine (XMAX, -idle_crow_limit, XMAX+10, -idle_crow_limit, SOLID)
      drawLine (XMAX, idle_crow_limit, XMAX+10, idle_crow_limit, SOLID)
    end

    -- draw vertical bar showing crow value
    drawLine (xSrc,YMIN, xSrc, YMAX, DOTTED)
  
    -- draw region separators
    for r = 1, #regions do
      drawLine (regions[r].xMax,-4, regions[r].xMax, 4, SOLID)
    end
    drawLine (XMIN,-4, XMIN, 4, SOLID)

    -- draw points
    for i=1, #pt_y do
      local ptsize = (is_blink and regions[rgn].pts[i]) and 2 or 1
      if IS_COLOR then 
          ptsize = ptsize + 1
      end 
      drawPoint (pt_x[i], pt_y[i], ptsize)
    end    
  end
   -- join the dots
  for i=1, #pt_y - 1 do
    drawLine (pt_x[i], pt_y[i], pt_x[i + 1],	pt_y[i + 1], fActive and SOLID or DOTTED)
  end
  
  	-- draw script name, bottom right
	lcd.drawText (xorg + width, yorg + height - line_ht,"v"..VERSION, font + INVERS + RIGHT)
end

local function refresh(widget)
    run_func (widget)
end

------------------
-- Helper funcs
------------------
function drawText (x, y, text, flags)
	lcd.drawText (x, y, text, font + (flags or 0))
end

-- Drawing funcs, use logical coordinates,
function drawLine (x1,y1, x2, y2, pattern)
	-- FORCE is nil for large screens
  lcd.drawLine (sx(x1), sy(y1), sx(x2), sy(y2), pattern, FORCE or 0)
end

function drawPoint (x, y, size)
  -- FORCE is nil for large screens
	lcd.drawFilledRectangle (sx(x)-size, sy(y)-size, size*2+1, size*2+1, FORCE or 0)
end

function sx (x)
  return (xorg + (XMAX - x) * x_scale)
end

function sy (y)
	return (yorg + (YMAX - y) * y_scale)
end

--[[
Wrapper for model.getCurve(), to handle both 0-based and 1-based internal y tables 
Outputs a curve table with 1-based y table.
(new in 3.2)
--]]
function getCurve(idx)
  local cv = model.getCurve (idx)
  if cv == nil then return nil end
  
  -- initialise copy
  local cvDest = {}
  cvDest.y = {}
  
  -- Base 0 or 1
  local j = cv.y[0] and 0 or 1
  
  -- copy and shift points
  for i = 1, cv.points do
    cvDest.y[i] = cv.y[j]
    j = j + 1
  end
  cvDest.smooth = cv.smooth
  cvDest.points = cv.points
  cvDest.name = cv.name
  cvDest.type = cv.type
  return cvDest
end

-- given the name of a curve, return the index.
function getCurveIdxFromName (name)
	local i = 0
	local cv
	while true do
		cv = model.getCurve (i)
		if not cv then break end
		if cv.name == name then
			return i
		end
		i = i + 1
	end
end

return IS_WGT and 
  {name="adptr"..VERSION, options=options, create=create, update=update, refresh=refresh, background=background } or
  {init=init, run=run_func, background=bg_func}
